/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { Environment } from './environment';
import { envSatisfiesConstraints, parseTestEnvFromKarmaFlags } from './jasmine_util';
describe('jasmine_util.envSatisfiesConstraints', () => {
    it('ENV satisfies empty constraints', () => {
        const env = new Environment({});
        env.setFlags({});
        const constraints = {};
        const backendName = 'test-backend';
        expect(envSatisfiesConstraints(env, { name: 'test', backendName }, constraints))
            .toBe(true);
    });
    it('ENV satisfies matching flag constraints no predicate', () => {
        const env = new Environment({});
        env.setFlags({ 'TEST-FLAG': true });
        const constraints = { flags: { 'TEST-FLAG': true } };
        const backendName = 'test-backend';
        expect(envSatisfiesConstraints(env, { name: 'test', backendName }, constraints))
            .toBe(true);
    });
    it('ENV satisfies matching flag and predicate is true', () => {
        const env = new Environment({});
        env.setFlags({ 'TEST-FLAG': true });
        const constraints = { flags: { 'TEST-FLAG': true }, predicate: () => true };
        const backendName = 'test-backend';
        expect(envSatisfiesConstraints(env, { name: 'test', backendName }, constraints))
            .toBe(true);
    });
    it('ENV doesnt satisfy flags and predicate is true', () => {
        const env = new Environment({});
        env.setFlags({ 'TEST-FLAG': true });
        const constraints = { flags: { 'TEST-FLAG': false }, predicate: () => true };
        const backendName = 'test-backend';
        expect(envSatisfiesConstraints(env, { name: 'test', backendName }, constraints))
            .toBe(false);
    });
    it('ENV satisfies flags and predicate is false', () => {
        const env = new Environment({});
        env.setFlags({ 'TEST-FLAG': true });
        const constraints = { flags: { 'TEST-FLAG': true }, predicate: () => false };
        const backendName = 'test-backend';
        expect(envSatisfiesConstraints(env, { name: 'test', backendName }, constraints))
            .toBe(false);
    });
    it('ENV doesnt satiisfy flags and predicate is false', () => {
        const env = new Environment({});
        env.setFlags({ 'TEST-FLAG': true });
        const constraints = { flags: { 'TEST-FLAG': false }, predicate: () => false };
        const backendName = 'test-backend';
        expect(envSatisfiesConstraints(env, { name: 'test', backendName }, constraints))
            .toBe(false);
    });
});
describe('jasmine_util.parseKarmaFlags', () => {
    const registeredTestEnvs = [
        { name: 'test-env', backendName: 'test-backend', isDataSync: true, flags: {} }
    ];
    it('parse empty args', () => {
        const res = parseTestEnvFromKarmaFlags([], registeredTestEnvs);
        expect(res).toBeNull();
    });
    it('--testEnv test-env --flags {"IS_NODE": true}', () => {
        const res = parseTestEnvFromKarmaFlags(['--testEnv', 'test-env', '--flags', '{"IS_NODE": true}'], registeredTestEnvs);
        expect(res.name).toBe('test-env');
        expect(res.backendName).toBe('test-backend');
        expect(res.flags).toEqual({ IS_NODE: true });
    });
    it('"--testEnv unknown" throws error', () => {
        expect(() => parseTestEnvFromKarmaFlags(['--testEnv', 'unknown'], registeredTestEnvs))
            .toThrowError();
    });
    it('"--flags {}" throws error since --testEnv is missing', () => {
        expect(() => parseTestEnvFromKarmaFlags(['--flags', '{}'], registeredTestEnvs))
            .toThrowError();
    });
    it('"--testEnv cpu --flags" throws error since features value is missing', () => {
        expect(() => parseTestEnvFromKarmaFlags(['--testEnv', 'test-env', '--flags'], registeredTestEnvs))
            .toThrowError();
    });
    it('"--backend cpu --flags notJson" throws error', () => {
        expect(() => parseTestEnvFromKarmaFlags(['--testEnv', 'test-env', '--flags', 'notJson'], registeredTestEnvs))
            .toThrowError();
    });
});
//# sourceMappingURL=data:application/json;base64,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